/* Analisador recursivo descendente de expressoes inteiras
   que pode incluir variveis e chamadas de funoes. */

#include <setjmp.h>
#include <math.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#define NUM_FUNC        100
#define NUM_GLOBAL_VARS 100
#define NUM_LOCAL_VARS  200
#define ID_LEN          31
#define FUNC_CALLS      31
#define PROG_SIZE       10000
#define FOR_NEST        31

enum tok_types {DELIMITER, IDENTIFIER, NUMBER, KEYWORD, TEMP,
                STRING, BLOCK};

enum tokens { ARG, CHAR, INT, IF, ELSE, FOR, DO, WHILE,
              SWITCH, RETURN, EOL, FINISHED, END };

enum double_ops { LT=1, LE, GT, GE, EQ, NE};

/* Estas sao constantes usadas para chamar sntx_err() quando
   ocorre um erro de sintaxe. Adicione mais se desejar.
   NOTA: SYNTAX  uma mensagem genrica de erro usada quando
   nenhuma outra parece apropriada.
 */
enum error_msg
     {SYNTAX, UNBAL_PARENS, NO_EXP, EQUALS_EXPECTED,
      NOT_VAR, PARAM_ERR, SEMI_EXPECTED,
      UNBAL_BRACES, FUNC_UNDEF, TYPE_EXPECTED,
      NEST_FUNC, RET_NOCALL, PAREN_EXPECTED,
      WHILE_EXPECTED, QUOTE_EXPECTED, NOT_TEMP,
      TOO_MANY_LVARS};

extern char *prog; /* posiao corrente no cdigo fonte */
extern char *p_buf; /* aponta para o incio da
                       rea de carga do programa */
extern jmp_buf e_buf; /* mantm ambiente para longjmp() */

/* Uma matriz destas estruturas manter a informaao
   associada com as variveis globais.
*/
extern struct var_type {
  char var_name[32];
  enum variable_type var_type;
  int value;
} global_vars[NUM_GLOBAL_VARS];

/* Esta  a pilha de chamadas de funao. */
extern struct func_type {
  char func_name[32];
  char *loc; /* posiao do ponto de entrada da funao no arquivo */
} func_stack[NUM_FUNC];

/* Tabela de palavras reservadas */
extern struct commands {
  char command[20];
  char tok;
} table[];

/* Funoes da "biblioteca padrao" sao declaradas aqui para que
   possam ser colocadas na tabela interna de funoes que segue.
 */
int call_getche(void), call_putch(void);
int call_puts(void), print(void), getnum(void);

struct intern_func_type {
  char *f_name; /* nome da funao */
  int (*p)();   /* ponteiro para a funao */
} intern_func[] = {
  "getche", call_getche,
  "putch", call_putch,
  "puts", call_puts,
  "print", print,
  "getnum", getnum,
  "", 0
};

extern char token[80]; /* representaao string do token */
extern char token_type; /* contm o tipo do token */
extern char tok; /* representaao interna do token */

extern int ret_value; /* valor de retorno de funao */

void eval_exp(int *value), eval_exp1(int *value);
void eval_exp2(int *value);
void eval_exp3(int *value), eval_exp4(int *value);
void eval_exp5(int *value), atom(int *value);
void eval_exp0(int *value);
void sntx_err(int error), putback(void);
void assign_var(char *var_name, int value);
int isdelim(char c), look_up(char *s), iswhite(char c);
int find_var(char *s), get_token(void);
int internal_func(char *s);
int is_var(char *s);
char *find_func(char *name);
void call(void);

/* Ponto de entrada do analisador. */
void eval_exp(int *value)
{
  get_token();
  if(!*token) {
    sntx_err(NO_EXP);
    return;
  }
  if(*token==';') {
    *value = 0; /* expressao vazia */
    return;
  }
  eval_exp0(value);
  putback(); /* devolve ltimo token lida para a entrada */
}

/* Processa uma expressao de atribuiao */
void eval_exp0(int *value)
{
  char temp[ID_LEN]; /* guarda nome da varivel que est
                        recebendo a atribuiao */
  register int temp_tok;

  if(token_type==IDENTIFIER) {
    if(is_var(token)) { /* se  uma varivel,
                           veja se  uma atribuiao */
      strcpy(temp, token);
      temp_tok = token_type;
      get_token();
      if(*token=='=') { /*  uma atribuiao */
        get_token();
        eval_exp0(value); /* obtm valor a atribuir */
        assign_var(temp, *value); /* atribui o valor */
        return;
      }
      else {  /* nao  uma atribuiao */
        putback(); /* restaura token original */
        strcpy(token, temp);
        token_type = temp_tok;
      }
    }
  }
  eval_exp1(value);
}

/* Esta matriz  usada por eval_exp1(). Como alguns
   compiladores nao permitem inicializar uma matriz
   dentro de uma funao, ela  definido como uma
   varivel global.
*/
char relops[7] = {
  LT, LE, GT, GE, EQ, NE, 0
};

/* Processa operadores relacionais */
void eval_exp1(int *value)
{
  int partial_value;
  register char op;

  eval_exp2(value);
  op = *token;
  if(strchr(relops, op)) {
    get_token();
    eval_exp2(&partial_value);
    switch(op) { /* efetua a operaao relacional */
      case LT:
        *value = *value < partial_value;
        break;
      case LE:
        *value = *value <= partial_value;
        break;
      case GT:
        *value = *value > partial_value;
        break;
      case GE:
        *value = *value >= partial_value;
        break;
      case EQ:
        *value = *value == partial_value;
        break;
      case NE:
        *value = *value != partial_value;
        break;
    }
  }
}

/* Soma ou subtrai dois termos. */
void eval_exp2(int *value)
{
  register char  op;
  int partial_value;

  eval_exp3(value);
  while((op = *token) == '+' || op == '-') {
    get_token();
    eval_exp3(&partial_value);
    switch (op) { /* soma ou subtrai */
      case '-':
        *value = *value - partial_value;
        break;
      case '+':
        *value = *value + partial_value;
        break;
    }
  }
}

/* Multiplica ou divide dois fatores. */
void eval_exp3(int *value)
{
  register char  op;
  int partial_value, t;

  eval_exp4(value);
  while((op = *token) == '*' || op == '/' || op == '%') {
    get_token();
    eval_exp4(value);
    switch(op) { /* mul, div ou mdulo */
      case '*':
        *value = *value * partial_value;
        break;
      case '/':
        *value = *value / partial_value;
        break;
      case '%':
        t = (*value) / partial_value;
        *value = *value-(t * partial_value);
        break;
    }
  }
}

/*  um + ou - unrio. */
void eval_exp4(int *value)
{
  register char  op;

  op = '\0';
  if(*token=='+' || *token=='-') {
    op = *token;
    get_token();
  }
  eval_exp5(value);
  if(op)
    if(op=='-') *value = -(*value);
}

/* Processa expressoes com parnteses. */
void eval_exp5(int *value)
{
  if((*token == '(')) {
    get_token();
    eval_exp0(value);  /* obtm subexpressao */
    if(*token != ')') sntx_err(PAREN_EXPECTED);
    get_token();
  }
  else
    atom(value);
}

/* Acha valor de nmero, varivel ou funao. */
void atom(int *value)
{
  int i;

  switch(token_type) {
  case IDENTIFIER:
    i = internal_func(token);
    if(i!= -1) { /* chama funao da "biblioteca padrao" */
      *value = (*intern_func[i].p)();
    }
    else
    if(find_func(token)){ /* chama funao definida pelo usurio */
      call();
      *value = ret_value;
    }
    else  *value = find_var(token);  /* obtm valor da varivel */
    get_token();
    return;
  case NUMBER: /*  uma constante numrica */
    *value = atoi(token);
    get_token();
    return;
  case DELIMITER: /* veja se  uma constante caractere */
    if(*token=='\'') {
      *value = *prog;
      prog++;
      if(*prog!='\'') sntx_err(QUOTE_EXPECTED);
      prog++;
      get_token();
    }
    return;
  default:
    if(*token==')') return; /* processa expressao vazia */
    else sntx_err(SYNTAX); /* erro de sintaxe */
  }
}

/* Exibe uma mensagem de erro. */
void sntx_err(int error)
{
  char *p, *temp;
  register int i;
  int linecount = 0;

  static char *e[] = {
    "erro de sintaxe",
    "parnteses desbalanceados",
    "falta uma expressao",
    "esperado sinal de igual",
    "nao  uma varivel",
    "erro de parmetro",
    "esperado ponto e vrgula",
    "chaves desbalanceadas",
    "funao nao definida",
    "esperado identificador de tipo",
    "excessivas chamadas aninhadas de funao",
    "return sem chamada",
    "esperado parnteses",
    "esperado while",
    "esperando fechar aspas",
    "nao  uma string",
    "excessivas variveis locais"
  };
  printf("%s", e[error]);
  p = p_buf;
  while(p != prog) { /* encontra linha do erro */
    p++;
    if(*p=='\r') {
      linecount++;
    }
  }
  printf(" na linha %d\n", linecount);

  temp = p; /* exibe linha contendo erro */
  for(i=0; i<20 && p>p_buf && *p!='\n'; i++, p--);
  for(i=0; i<30 && p<=temp; i++, p++) printf("%c", *p);

  longjmp(e_buf, 1); /* retorna para um ponto seguro */
}

/* Obtm um token. */
get_token(void)
{
  register char *temp;

  token_type = 0; tok = 0;

  temp = token;
  *temp = '\0';

  /* ignora espao vazio */
  while(iswhite(*prog) && *prog) ++prog;

  if(*prog=='\r') {
    ++prog;
    ++prog;
    /* ignora espao vazio */
    while(iswhite(*prog) && *prog) ++prog;
  }

  if(*prog=='\0') { /* fim de arquivo */
    *token = '\0';
    tok = FINISHED;
    return(token_type=DELIMITER);
  }

  if(strchr("{}", *prog)) { /* delimitadores de bloco */
    *temp = *prog;
    temp++;
    *temp = '\0';
    prog++;
    return (token_type = BLOCK);
  }

  /* procura por comentrios */
  if(*prog=='/')
    if(*(prog+1)=='*') { /*  um comentrio */
      prog += 2;
      do { /* procura fim do comentrio */
        while(*prog!='*') prog++;
        prog++;
      } while (*prog!='/');
      prog++;
    }

  if(strchr("!<>=", *prog)) { /*  ou pode ser
                              um operador relacional */
    switch(*prog) {
      case '=': if(*(prog+1)=='=') {
          prog++; prog++;
          *temp = EQ;
          temp++; *temp = EQ; temp++;
          *temp = '\0';
        }
        break;
      case '!': if(*(prog+1)=='=') {
          prog++; prog++;
          *temp = NE;
          temp++; *temp = NE; temp++;
          *temp = '\0';
        }
        break;
      case '<': if(*(prog+1)=='=') {
          prog++; prog++;
          *temp = LE; temp++; *temp = LE;
        }
        else {
          prog++;
          *temp = LT;
        }
        temp++;
        *temp = '\0';
        break;
      case '>': if(*(prog+1)=='=') {
          prog++; prog++;
          *temp = GE; temp++; *temp = GE;
        }
        else {
          prog++;
          *temp = GT;
        }
        temp++;
        *temp = '\0';
        break;
    }
    if(*token) return(token_type = DELIMITER);
  }

  if(strchr("+-*^/%=;(),'", *prog)){ /* delimitador */
    *temp = *prog;
    prog++; /* avana para a prxima posiao */
    temp++;
    *temp = '\0';
    return (token_type = DELIMITER);
  }

  if(*prog=='"') { /* string entre aspas */
    prog++;
    while(*prog!='"'&& *prog!='\r') *temp++=*prog++;
    if(*prog=='\r') sntx_err(SYNTAX);
    prog++; *temp = '\0';
    return(token_type=STRING);
  }

  if(isdigit(*prog)) { /* nmero */
    while(!isdelim(*prog)) *temp++ = *prog++;
    *temp = '\0';
    return(token_type = NUMBER);
  }

  if (isalpha(*prog)) { /* varivel ou comando */
    while(!isdelim(*prog)) *temp++ = *prog++;
    token_type=TEMP;
  }

  *temp = '\0';

  /* verifica se uma string  um comando ou uma varivel */
  if(token_type==TEMP) {
    tok = look_up(token); /* converte para representaao interna */
    if(tok) token_type = KEYWORD; /*  uma palavra reservada */
    else token_type = IDENTIFIER;
  }
  return token_type;
}

/* Devolve um token para a entrada. */
void putback(void)
{
  char *t;

  t = token;
  for(; *t; t++) prog--;
}

/* Procura pela representaao interna de um token na
   tabela de tokens.
*/
look_up(char *s)
{
  register int i;
  char *p;

  /* converte para minscula */
  p = s;
  while(*p){ *p = tolower(*p); p++; }

  /* verifica se o token est na tabela */
  for(i=0; *table[i].command; i++)
    if(!strcmp(table[i].command, s)) return table[i].tok;
  return 0; /* comando desconhecido */
}

/* Retorna indce de funao da biblioteca interna ou -1
   se nao encontrada.
*/
internal_func(char *s)
{
  int i;

  for(i=0; intern_func[i].f_name[0]; i++) {
    if(!strcmp(intern_func[i].f_name, s)) return i;
  }
  return -1;
}

/* Retorna verdadeiro se c  um delimitador. */
isdelim(char c)
{
  if(strchr(" !;,+-<>'/*%^=()", c) || c==9 ||
     c=='\r' || c==0) return 1;
  return 0;
}

/* Retorna 1 se c  espao ou tabulaao. */
iswhite(char c)
{
  if(c==' ' || c=='\t') return 1;
  else return 0;
}

